import 'dart:io';

import 'package:admin_desktop/domain/models/models.dart';
import 'package:admin_desktop/domain/models/response/sale_history_response.dart';
import 'package:admin_desktop/presentation/components/custom_line.dart';
import 'package:flutter/material.dart';
import 'package:flutter_esc_pos_utils/flutter_esc_pos_utils.dart';
import 'package:flutter_pos_printer_platform_image_3/flutter_pos_printer_platform_image_3.dart';
import 'package:flutter_screenutil/flutter_screenutil.dart';
import 'package:admin_desktop/infrastructure/services/utils.dart';
import 'package:admin_desktop/presentation/components/components.dart';
import 'package:remixicon_updated/remixicon_updated.dart';
import 'package:admin_desktop/presentation/components/buttons/custom_mini_button.dart';
import 'package:admin_desktop/presentation/styles/style.dart';
import '../order_detail/print_page.dart';

class GenerateSaleHistoryCheck extends StatefulWidget {
  final SaleCartResponse? saleHistory;
  final List<SaleHistoryModel> list;

  const GenerateSaleHistoryCheck(
      {super.key, required this.saleHistory, required this.list});

  @override
  State<GenerateSaleHistoryCheck> createState() => _GenerateSaleHistoryCheckState();
}

class _GenerateSaleHistoryCheckState extends State<GenerateSaleHistoryCheck> {
  @override
  Widget build(BuildContext context) {
    return Container(
      decoration: BoxDecoration(
        color: Style.white,
        borderRadius: BorderRadius.circular(10),
      ),
      padding: REdgeInsets.symmetric(horizontal: 14, vertical: 16),
      child: SingleChildScrollView(
        child: Column(
          mainAxisSize: MainAxisSize.min,
          crossAxisAlignment: CrossAxisAlignment.center,
          children: [
            Text(AppHelpers.getTranslation(TrKeys.saleHistory),
                style: Style.interMedium(size: 20)),
            Text(
              TimeService.dateFormatYMDHm(DateTime.now()),
              style: Style.interNormal(size: 12),
            ),
            12.verticalSpace,
            CustomLine(),
            8.verticalSpace,
            Row(
              children: [
                SizedBox(
                  width: 120.w,
                  child: Text(
                      AppHelpers.getTranslation(TrKeys.shopName),
                      style: Style.interRegular(size: 14)),
                ),
                const Spacer(),
                Flexible(
                  child: Text(LocalStorage.getShop()?.translation?.title ?? '',
                      style: Style.interRegular(size: 14),textAlign: TextAlign.end,),
                )
              ],
            ),
            12.verticalSpace,
            Row(
              children: [
                SizedBox(
                  width: 120.w,
                  child: Text(
                      AppHelpers.getTranslation(TrKeys.openingDrawerAmount),
                      style: Style.interRegular(size: 14)),
                ),
                const Spacer(),
                Text(AppHelpers.numberFormat(number: widget.saleHistory?.deliveryFee),
                    style: Style.interRegular(size: 14))
              ],
            ),
            8.verticalSpace,

            Row(
              children: [
                Expanded(
                  child: Text(AppHelpers.getTranslation(TrKeys.cashPaymentSale),
                      style: Style.interRegular(size: 14)),
                ),
                const Spacer(),
                Text(AppHelpers.numberFormat(number: widget.saleHistory?.cash),
                    style: Style.interRegular(size: 14))
              ],
            ),
            Column(
              children: [
                10.verticalSpace,
                Row(
                  children: [
                    SizedBox(
                      width: 120.w,
                      child: Text(
                        AppHelpers.getTranslation(TrKeys.otherPaymentSale),
                        style: Style.interRegular(size: 14),
                      ),
                    ),
                    const Spacer(),
                    Text(
                      AppHelpers.numberFormat(number: widget.saleHistory?.other),
                      style: Style.interRegular(size: 14),
                    )
                  ],
                ),
              ],
            ),
            12.verticalSpace,
            Divider(),
          widget.list.isNotEmpty ? Column(
              children: [
                CustomLine(),
                12.verticalSpace,
                Table(
                  columnWidths: {
                    0: FixedColumnWidth(50.0),
                    1: FixedColumnWidth(150.0),
                    2: FixedColumnWidth(100),
                  },
                  children: [
                    TableRow(children: [
                      Text(
                        AppHelpers.getTranslation(TrKeys.id),
                        style: Style.interNormal(size: 14),
                      ),
                      Text(
                        AppHelpers.getTranslation(TrKeys.name),
                        style: Style.interNormal(size: 14),
                      ),
                      Text(
                        AppHelpers.getTranslation(TrKeys.price),
                        style: Style.interNormal(size: 14),
                      ),
                    ]),
                  ],
                ),
                10.verticalSpace,
                CustomLine(),
                10.verticalSpace,
                Table(
                    columnWidths: {
                      0: FixedColumnWidth(50.0),
                      1: FixedColumnWidth(150.0),
                      2: FixedColumnWidth(100),
                    },
                    children: widget.list.map((data) {
                      return TableRow(children: [
                        Text(
                          "${data.id ?? 0}",
                          style: Style.interRegular(size: 14),
                        ),
                        Text(
                          data.user?.firstname ?? "",
                          style: Style.interRegular(size: 14),
                        ),
                        Text(
                          AppHelpers.numberFormat(number: data.totalPrice),
                          style: Style.interRegular(size: 14),
                        ),
                      ]);
                    }).toList()),
              ],
            ) : SizedBox.shrink(),

            26.verticalSpace,
            Row(
              mainAxisAlignment: MainAxisAlignment.spaceBetween,
              children: [
                Expanded(
                  child: LoginButton(
                      title: AppHelpers.getTranslation(TrKeys.print),
                      onPressed: () async {
                        if(LocalStorage.getPrinterData()==null){
                          final bytes = await _getBytes();
                          if (context.mounted) {
                            AppHelpers.showAlertDialog(
                                context: context, child: PrintPage(bytes: bytes));
                          }
                        }else {
                          final bytes = await _getBytes();
                          _printEscPos(
                            bytes,
                          );
                        }
                      }),
                ),
                10.horizontalSpace,
                CustomMiniButton(
                  size: 56,
                  onTap: () async {
                    final bytes = await _getBytes();
                    if (context.mounted) {
                      AppHelpers.showAlertDialog(
                          context: context, child: PrintPage(bytes: bytes));
                    }
                    // ignore: deprecated_member_use
                  },
                  icon: Remix.settings_5_fill,
                  iconColor: Style.white.withOpacity(0.7),
                  color: Style.primary,
                )
              ],
            ),
          ],
        ),
      ),
    );
  }

  var printerManager = PrinterManager.instance;

  List<int>? pendingTask;

  final _reconnect = false;

  final BTStatus _currentStatus = BTStatus.none;

  void _printEscPos(List<int> bytes,) async {
    final profile = await CapabilityProfile.load();
    final generator = Generator(PaperSize.mm58, profile);
    var connectedTCP = false;
    if (LocalStorage.getPrinterData() == null) return;
    var bluetoothPrinter = LocalStorage.getPrinterData()!;

    switch (bluetoothPrinter.typePrinter) {
      case PrinterType.usb:
        debugPrint("");
        bytes += generator.feed(2);
        bytes += generator.cut();
        await printerManager.connect(
            type: bluetoothPrinter.typePrinter,
            model: UsbPrinterInput(
                name: bluetoothPrinter.deviceName,
                productId: bluetoothPrinter.productId,
                vendorId: bluetoothPrinter.vendorId));
        pendingTask = null;
        break;
      case PrinterType.bluetooth:
        bytes += generator.cut();
        await printerManager.connect(
            type: bluetoothPrinter.typePrinter,
            model: BluetoothPrinterInput(
                name: bluetoothPrinter.deviceName,
                address: bluetoothPrinter.address!,
                isBle: bluetoothPrinter.isBle ?? false,
                autoConnect: _reconnect));
        pendingTask = null;
        if (Platform.isAndroid) pendingTask = bytes;
        break;
      case PrinterType.network:
        bytes += generator.feed(2);
        bytes += generator.cut();
        connectedTCP = await printerManager.connect(
            type: bluetoothPrinter.typePrinter,
            model: TcpPrinterInput(ipAddress: bluetoothPrinter.address!));
        if (!connectedTCP) debugPrint(' --- please review your connection ---');
        break;
    }
    if (bluetoothPrinter.typePrinter == PrinterType.bluetooth &&
        Platform.isAndroid) {
      if (_currentStatus == BTStatus.connected) {
        printerManager.send(type: bluetoothPrinter.typePrinter, bytes: bytes);
        pendingTask = null;
      }
    } else {
      printerManager.send(type: bluetoothPrinter.typePrinter, bytes: bytes);
    }
  }

  Future<List<int>> _getBytes() async {
    List<int> bytes = [];
    String customDivider = "-------------------------------------------";
    String customLine = "___________________________________________";
    String customSpace = "  ";
    final profile = await CapabilityProfile.load(name: 'XP-N160I');

    final generator = Generator(PaperSize.mm80, profile);

    bytes += generator.setGlobalCodeTable('CP1252');
    bytes += generator.setStyles(const PosStyles(align: PosAlign.center));
    bytes += generator.text(AppHelpers.getTranslation(TrKeys.saleHistory),
        styles: const PosStyles(align: PosAlign.center, bold: true));

    bytes += generator.text(TimeService.dateFormatYMDHm(DateTime.now()),
        styles: const PosStyles(align: PosAlign.center));
    bytes += generator.text(customDivider);
    bytes += generator.row([
      PosColumn(
          width: 7,
          text: customSpace +
              AppHelpers.getTranslation(TrKeys.openingDrawerAmount),
          styles: const PosStyles(align: PosAlign.left, codeTable: 'CP1252')),
      PosColumn(
          width: 5,
          text: AppHelpers.numberFormat(number: widget.saleHistory?.deliveryFee),
          styles: const PosStyles(align: PosAlign.right, codeTable: 'CP1252')),
    ]);
    bytes += generator.text(
    'привет',
      containsChinese: true,
      styles: const PosStyles(align: PosAlign.center, bold: true),
    );
    bytes += generator.row([
      PosColumn(
        width: 6,
        text: customSpace + AppHelpers.getTranslation(TrKeys.cashPaymentSale),
        styles: const PosStyles(align: PosAlign.left),
      ),
      PosColumn(
        width: 6,
        text: AppHelpers.numberFormat(number: widget.saleHistory?.cash),
        styles: const PosStyles(align: PosAlign.right),
      ),
    ]);

    bytes += generator.row([
      PosColumn(
        width: 6,
        text: customSpace + AppHelpers.getTranslation(TrKeys.average),
        styles: const PosStyles(
          align: PosAlign.left,
        ),
      ),
      PosColumn(
        width: 6,
        text: AppHelpers.numberFormat(number: widget.saleHistory?.other),
        styles: const PosStyles(
          align: PosAlign.right,
        ),
      ),
    ]);
    bytes += generator.text(customLine);
    bytes += generator.text(customDivider);
    bytes += generator.row([
      PosColumn(
          text: customSpace+TrKeys.id,
          width: 2,
          styles: PosStyles(bold: true, align: PosAlign.left)),
      PosColumn(
          text: TrKeys.name,
          width: 6,
          styles: PosStyles(bold: true, align: PosAlign.left)),
      PosColumn(
          text: TrKeys.price,
          width: 4,
          styles: PosStyles(bold: true, align: PosAlign.right)),
    ]);
    bytes += generator.text(customDivider);
    for (var data in widget.list) {
      bytes += generator.row([
        PosColumn(
            text: '$customSpace${data.id ?? 0}',
            width: 2,
            styles: PosStyles(align: PosAlign.left)),
        PosColumn(
            text: data.user?.firstname ?? "",
            width: 6,
            styles: PosStyles(align: PosAlign.left)),
        PosColumn(
            text: AppHelpers.numberFormat(number: data.totalPrice),
            width: 4,
            styles: PosStyles(align: PosAlign.right)),
      ]);
    }

    return bytes;
  }
}
